untyped

global function Score_Init

global function AddPlayerScore
global function ScoreEvent_PlayerKilled
global function ScoreEvent_TitanDoomed
global function ScoreEvent_TitanKilled
global function ScoreEvent_NPCKilled
global function ScoreEvent_MatchComplete
global function ScoreEvent_RoundComplete

global function ScoreEvent_SetEarnMeterValues
global function ScoreEvent_SetupEarnMeterValuesForMixedModes
global function ScoreEvent_SetupEarnMeterValuesForTitanModes

struct {
	bool firstStrikeDone = false
} file

void function Score_Init()
{
	SvXP_Init()
	AddCallback_OnClientConnected( InitPlayerForScoreEvents )

	AddCallback_OnPlayerAssist( TitanAssistedKill )
	ScoreEvent_SetDisplayType( GetScoreEvent( "KillingSpree" ), eEventDisplayType.GAMEMODE | eEventDisplayType.MEDAL | eEventDisplayType.CALLINGCARD )
	ScoreEvent_SetDisplayType( GetScoreEvent( "Rampage" ), eEventDisplayType.GAMEMODE | eEventDisplayType.MEDAL | eEventDisplayType.CALLINGCARD )
}

void function InitPlayerForScoreEvents( entity player )
{	
	player.s.currentKillstreak <- 0
	player.s.lastKillTime <- 0.0
	player.s.currentTimedKillstreak <- 0
	player.s.lastKillTime_Mayhem <- 0.0
	player.s.currentTimedKillstreak_Mayhem <- 0 
	player.s.lastKillTime_Onslaught <- 0.0
	player.s.currentTimedKillstreak_Onslaught <- 0 
}

void function AddPlayerScore( entity targetPlayer, string scoreEventName, entity associatedEnt = null, string noideawhatthisis = "", int pointValueOverride = -1 )
{
	ScoreEvent event = GetScoreEvent( scoreEventName )
	
	if ( !event.enabled || !IsValidPlayer( targetPlayer ) )
		return

	var associatedHandle = 0
	if ( associatedEnt != null )
		associatedHandle = associatedEnt.GetEncodedEHandle()
	
	if ( pointValueOverride != -1 )
		event.pointValue = pointValueOverride 
	
	float scale = targetPlayer.IsTitan() ? event.coreMeterScalar : 1.0
	
	float earnValue = event.earnMeterEarnValue * scale
	float ownValue = event.earnMeterOwnValue * scale
	
	if ( !PlayerEarnMeter_Enabled() && !targetPlayer.IsTitan() ) // Don't show earning points if earn meter is not enabled and player is not a special case of being a titan
	{
		earnValue = 0.0
		ownValue = 0.0
	}

	// Both checks below are mostly a visual fix because the score medals would still show the adds into the total value
	if ( PlayerEarnMeter_GetPilotOverdriveEnum() == ePilotOverdrive.Disabled )
		earnValue = 0.0
	
	if ( PlayerEarnMeter_GetPilotOverdriveEnum() == ePilotOverdrive.Only )
		ownValue = 0.0
	
	PlayerEarnMeter_AddEarnedAndOwned( targetPlayer, earnValue * scale, ownValue * scale )
	SharedEarnMeter_AddEarnedAndOwned( targetPlayer, earnValue, ownValue )
	
	// PlayerEarnMeter_AddEarnedAndOwned handles this scaling by itself, we just need to do this for the visual stuff
	float pilotScaleVar = ( expect string ( GetCurrentPlaylistVarOrUseValue( "earn_meter_pilot_multiplier", "1" ) ) ).tofloat()
	float titanScaleVar = ( expect string ( GetCurrentPlaylistVarOrUseValue( "earn_meter_titan_multiplier", "1" ) ) ).tofloat()
	
	if ( targetPlayer.IsTitan() )
	{
		if ( targetPlayer.GetPlayerNetInt( EARNMETER_MODE ) == eEarnMeterMode.CORE_ACTIVE ) // While core is active, Titans can't gain meter
		{
			earnValue = 0.0
			ownValue = 0.0
		}
		else
		{
			earnValue *= titanScaleVar
			ownValue *= titanScaleVar
		}
	}
	else
	{
		earnValue *= pilotScaleVar
		ownValue *= pilotScaleVar
	}
	
	Remote_CallFunction_NonReplay( targetPlayer, "ServerCallback_ScoreEvent", event.eventId, event.pointValue, event.displayType, associatedHandle, earnValue, ownValue )
	
	if ( event.displayType & eEventDisplayType.CALLINGCARD ) // callingcardevents are shown to all players
	{
		foreach ( entity player in GetPlayerArray() )
		{
			if ( player == targetPlayer ) // targetplayer already gets this in the scorevent callback
				continue
				
			Remote_CallFunction_NonReplay( player, "ServerCallback_CallingCardEvent", event.eventId, associatedHandle )
		}
	}
	
	if ( ScoreEvent_HasConversation( event ) )
		PlayFactionDialogueToPlayer( event.conversation, targetPlayer )
		
	HandleXPGainForScoreEvent( targetPlayer, event )
}

void function ScoreEvent_PlayerKilled( entity victim, entity attacker, var damageInfo )
{
	// reset killstreaks and stuff		
	victim.s.currentKillstreak = 0
	victim.s.lastKillTime = 0.0
	victim.s.currentTimedKillstreak = 0
	
	victim.p.numberOfDeathsSinceLastKill++ // this is reset on kill
	victim.p.lastKiller = attacker
	
	// have to do this early before we reset victim's player killstreaks
	// nemesis when you kill a player that is dominating you
	if ( attacker.IsPlayer() && attacker in victim.p.playerKillStreaks && victim.p.playerKillStreaks[ attacker ] >= NEMESIS_KILL_REQUIREMENT )
		AddPlayerScore( attacker, "Nemesis" )
	
	// reset killstreaks on specific players
	foreach ( entity killstreakPlayer, int numKills in victim.p.playerKillStreaks )
		delete victim.p.playerKillStreaks[ killstreakPlayer ]

	if ( victim.IsTitan() )
		ScoreEvent_TitanKilled( victim, attacker, damageInfo )

	if ( !attacker.IsPlayer() )
		return

	attacker.p.numberOfDeathsSinceLastKill = 0 // since they got a kill, remove the comeback trigger
	// pilot kill
	AddPlayerScore( attacker, "KillPilot", victim )
	
	// headshot
	if ( DamageInfo_GetCustomDamageType( damageInfo ) & DF_HEADSHOT )
		AddPlayerScore( attacker, "Headshot", victim )
	
	// first strike
	if ( !file.firstStrikeDone )
	{
		file.firstStrikeDone = true
		AddPlayerScore( attacker, "FirstStrike", attacker )
	}
	
	// comeback
	if ( attacker.p.numberOfDeathsSinceLastKill >= COMEBACK_DEATHS_REQUIREMENT )
	{
		AddPlayerScore( attacker, "Comeback" )
		attacker.p.numberOfDeathsSinceLastKill = 0
	}
	
	// revenge + quick revenge
	if ( attacker.p.lastKiller == victim )
	{
		if ( Time() - GetPlayerLastRespawnTime( attacker ) < QUICK_REVENGE_TIME_LIMIT )
			AddPlayerScore( attacker, "QuickRevenge" )
		else
			AddPlayerScore( attacker, "Revenge" )
	}
	
	// untimed killstreaks
	attacker.s.currentKillstreak++
	if ( attacker.s.currentKillstreak == KILLINGSPREE_KILL_REQUIREMENT )
		AddPlayerScore( attacker, "KillingSpree", attacker )
	else if ( attacker.s.currentKillstreak == RAMPAGE_KILL_REQUIREMENT )
		AddPlayerScore( attacker, "Rampage", attacker )
	
	// increment untimed killstreaks against specific players
	if ( !( victim in attacker.p.playerKillStreaks ) )
		attacker.p.playerKillStreaks[ victim ] <- 1
	else
		attacker.p.playerKillStreaks[ victim ]++
	
	// dominating
	if ( attacker.p.playerKillStreaks[ victim ] >= DOMINATING_KILL_REQUIREMENT )
		AddPlayerScore( attacker, "Dominating" )
	
	if ( Time() - attacker.s.lastKillTime > CASCADINGKILL_REQUIREMENT_TIME )
	{
		attacker.s.currentTimedKillstreak = 0 // reset first before kill
		attacker.s.lastKillTime = Time()
	}
	
	// timed killstreaks
	if ( Time() - attacker.s.lastKillTime <= CASCADINGKILL_REQUIREMENT_TIME )
	{
		attacker.s.currentTimedKillstreak++
		
		if ( attacker.s.currentTimedKillstreak == DOUBLEKILL_REQUIREMENT_KILLS )
			AddPlayerScore( attacker, "DoubleKill" )
		else if ( attacker.s.currentTimedKillstreak == TRIPLEKILL_REQUIREMENT_KILLS )
			AddPlayerScore( attacker, "TripleKill" )
		else if ( attacker.s.currentTimedKillstreak >= MEGAKILL_REQUIREMENT_KILLS )
			AddPlayerScore( attacker, "MegaKill", attacker )
	}
	
	attacker.s.lastKillTime = Time()
}

void function ScoreEvent_TitanDoomed( entity titan, entity attacker, var damageInfo )
{
	// will this handle npc titans with no owners well? i have literally no idea
	
	if ( titan.IsNPC() )
		AddPlayerScore( attacker, "DoomAutoTitan", titan )
	else
		AddPlayerScore( attacker, "DoomTitan", titan )
}

void function ScoreEvent_TitanKilled( entity victim, entity attacker, var damageInfo )
{
	// will this handle npc titans with no owners well? i have literally no idea
	if ( !attacker.IsPlayer() )
		return

	if ( attacker.IsTitan() )
	{
		if ( victim.GetBossPlayer() || victim.IsPlayer() ) // to confirm this is a pet titan or player titan
			AddPlayerScore( attacker, "TitanKillTitan", attacker ) // this will show the "Titan Kill" callsign event
		else
			AddPlayerScore( attacker, "TitanKillTitan" )
	}
	else
	{
		KilledPlayerTitanDialogue( attacker, victim )
		if ( victim.GetBossPlayer() || victim.IsPlayer() )
			AddPlayerScore( attacker, "KillTitan", attacker )
		else
			AddPlayerScore( attacker, "KillTitan" )
	}
}

void function TitanAssistedKill( entity attacker, entity victim )
{
	if ( IsSoul( victim ) )
	{
		AddPlayerScore( attacker, "TitanAssist" )
		attacker.AddToPlayerGameStat( PGS_ASSISTS, 1 )
	}
}

void function ScoreEvent_NPCKilled( entity victim, entity attacker, var damageInfo )
{
	if ( !attacker.IsPlayer() )
		return
	
	if ( DamageInfo_GetCustomDamageType( damageInfo ) & DF_HEADSHOT )
		AddPlayerScore( attacker, "NPCHeadshot" )
	
	try
	{		
		// have to trycatch this because marvins will crash on kill if we dont
		AddPlayerScore( attacker, ScoreEventForNPCKilled( victim, damageInfo ), victim )
	}
	catch ( ex ) {}
	
	// mayhem/onslaught (timed killstreaks vs AI)
	
	// reset before checking
	if ( Time() - attacker.s.lastKillTime_Mayhem > MAYHEM_REQUIREMENT_TIME )
	{
		attacker.s.currentTimedKillstreak_Mayhem = 0
		attacker.s.lastKillTime_Mayhem = Time()
	}
	if ( Time() - attacker.s.lastKillTime_Mayhem <= MAYHEM_REQUIREMENT_TIME )
	{
		attacker.s.currentTimedKillstreak_Mayhem++
		
		if ( attacker.s.currentTimedKillstreak_Mayhem == MAYHEM_REQUIREMENT_KILLS )
			AddPlayerScore( attacker, "Mayhem" )
	}

	// reset before checking
	if ( Time() - attacker.s.lastKillTime_Onslaught > ONSLAUGHT_REQUIREMENT_TIME )
	{
		attacker.s.currentTimedKillstreak_Onslaught = 0
		attacker.s.lastKillTime_Onslaught = Time()
	}
	if ( Time() - attacker.s.lastKillTime_Onslaught <= ONSLAUGHT_REQUIREMENT_TIME )
	{
		attacker.s.currentTimedKillstreak_Onslaught++
		
		if ( attacker.s.currentTimedKillstreak_Onslaught == ONSLAUGHT_REQUIREMENT_KILLS )
			AddPlayerScore( attacker, "Onslaught" )
	}
}

void function ScoreEvent_MatchComplete( int winningTeam )
{
	foreach( entity player in GetPlayerArray() )
	{
		AddPlayerScore( player, "MatchComplete" )
		SetPlayerChallengeMatchComplete( player )
		if ( player.GetTeam() == winningTeam )
		{
			AddPlayerScore( player, "MatchVictory" )
			SetPlayerChallengeMatchWon( player, true )
		}
		else
			SetPlayerChallengeMatchWon( player, false )
	}
}

void function ScoreEvent_RoundComplete( int winningTeam )
{
	foreach( entity player in GetPlayerArray() )
	{
		AddPlayerScore( player, "RoundComplete" )
		if ( player.GetTeam() == winningTeam )
			AddPlayerScore( player, "RoundVictory" )
	}
}

void function ScoreEvent_SetEarnMeterValues( string eventName, float earned, float owned, float coreScale = 1.0 )
{
	ScoreEvent event = GetScoreEvent( eventName )
	event.earnMeterEarnValue = earned
	event.earnMeterOwnValue = owned
	event.coreMeterScalar = coreScale
}

void function ScoreEvent_SetupEarnMeterValuesForMixedModes() // mixed modes in this case means modes with both pilots and titans
{
	// todo needs earn/overdrive values
	// player-controlled stuff
	ScoreEvent_SetEarnMeterValues( "KillPilot", 0.07, 0.15, 0.33 )
	ScoreEvent_SetEarnMeterValues( "KillTitan", 0.0, 0.15 )
	ScoreEvent_SetEarnMeterValues( "TitanKillTitan", 0.0, 0.0 ) // unsure
	ScoreEvent_SetEarnMeterValues( "PilotBatteryStolen", 0.0, 0.35 ) // this actually just doesn't have overdrive in vanilla even
	ScoreEvent_SetEarnMeterValues( "Headshot", 0.0, 0.02 )
	ScoreEvent_SetEarnMeterValues( "FirstStrike", 0.0, 0.05 )
	ScoreEvent_SetEarnMeterValues( "PilotBatteryApplied", 0.0, 0.35 )
	
	// ai
	ScoreEvent_SetEarnMeterValues( "KillGrunt", 0.02, 0.02, 0.5 )
	ScoreEvent_SetEarnMeterValues( "KillSpectre", 0.02, 0.02, 0.5 )
	ScoreEvent_SetEarnMeterValues( "LeechSpectre", 0.02, 0.02 )
	ScoreEvent_SetEarnMeterValues( "KillStalker", 0.02, 0.02, 0.5 )
	ScoreEvent_SetEarnMeterValues( "KillSuperSpectre", 0.0, 0.1, 0.5 )
}

void function ScoreEvent_SetupEarnMeterValuesForTitanModes()
{
	// relatively sure we don't have to do anything here but leaving this function for consistency
}

// faction dialogue
void function KilledPlayerTitanDialogue( entity attacker, entity victim )
{
	if ( !IsValidPlayer( attacker ) )
		return
	
	if ( !IsValid( victim ) || !victim.IsTitan() )
		return
	
	switch( GetTitanCharacterName( victim ) )
	{
		case "ion":
			PlayFactionDialogueToPlayer( "kc_pilotkillIon", attacker )
			return
		case "tone":
			PlayFactionDialogueToPlayer( "kc_pilotkillTone", attacker )
			return
		case "legion":
			PlayFactionDialogueToPlayer( "kc_pilotkillLegion", attacker )
			return
		case "scorch":
			PlayFactionDialogueToPlayer( "kc_pilotkillScorch", attacker )
			return
		case "ronin":
			PlayFactionDialogueToPlayer( "kc_pilotkillRonin", attacker )
			return
		case "northstar":
			PlayFactionDialogueToPlayer( "kc_pilotkillNorthstar", attacker )
			return
		default:
			PlayFactionDialogueToPlayer( "kc_pilotkilltitan", attacker )
			return
	}
}
